from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector, \
    get_var_assignment_matches_containing_url_alias, \
    contains_alias
from codeable_detectors.evidences import FailedEvidence, LinkEvidence
from codeable_detectors.php.phpDetectors import get_php_variable_assignments
from codeable_detectors.pyparsing_patterns import round_braces_block, ID
from codeable_detectors.utils import get_required_keyword_arg


class PHPCurlLink(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["php"]
        self.curl_init_pattern = Literal("$") + ID + Literal("=") + Literal("curl_init") + round_braces_block + \
                                 Literal(";")

    def detect_in_context(self, ctx, **kwargs):
        target = get_required_keyword_arg('target', kwargs)
        matches = []

        curl_init_matches = ctx.matches_pattern(self.curl_init_pattern)
        if not curl_init_matches:
            return FailedEvidence("curl init invocation not found")

        var_assignments_matching_aliases = get_var_assignment_matches_containing_url_alias(
            get_php_variable_assignments(ctx),
            target.aliases)

        for curl_init_match in curl_init_matches:
            parameters = curl_init_match.text[curl_init_match.text.find("(") + 1:curl_init_match.text.find(")")]
            if contains_alias(parameters, target.aliases):
                matches.append(curl_init_match)
            else:
                # maybe a variable in the HTTP post method contains the url to the alias
                for var, var_assignment_match in var_assignments_matching_aliases:
                    if var in parameters:
                        matches.append(curl_init_match)
                        matches.append(var_assignment_match)

        if matches:
            return LinkEvidence(matches).set_properties(detector_link_types="restfulHTTP",
                                                        detector_technology_types=["php", "curl"], kwargs=kwargs)
        return FailedEvidence("PHP curl init invocation to target not found")
